	% Example to demonstrate the scalability of the generator.
	% The example filter is a low-pass filter generated with 
	% firpmord([0.4 0.6],[1 0],[0.01 0.01],2)
	% The coefficients are scaled with 2^12 and rounded.
	% Note that the TF structure generates a 5us long transient in the 
	% simulation.
	
	
	% Parameter section
	
	% Filter impulse response
	% (negative coefficients _should_ work, but there are likely bugs,
	% verify generated filter with testbench)
	% Generate filter of order 234:

	[N,Fo,Ao,W] = firpmord([0.49 0.51],[1 0],[0.1 0.0001],2);
	h = round(2^16*firpm(N,Fo,Ao,W));
	
	% Data wordlength
	wdata = 16;
	% Signedness of data and coeffs: 0 for unsigned, 1 for signed two's 
	% complement. Signed coefficients require signed data.
	signedcoeffs = 1;
	signeddata = 1;
	
	% Coefficient type. Possible choices are 'bin' for binary and 'msd' for 
	% minimum signed digit. 
	ctype = 'msd';
	% For msd, all representations are indexed by cid.
	cid = 0;
	
	% Architecture specification. type can be 'TF' for transposed direct form,
	% or 'DF' for direct form. maxheight is the maximum adder depth.
	archspec.type = 'TF';
	archspec.maxheight = 2;
	archspec.symmetrysize = wdata;
	
	% Directory to generate the output files in
	vhdldir = 'vhdl_ex4';
	% File name prefix of generated files
	vhdlname = 'fir';
	
	
	% Code section (there should be no need to change anything below)
	
	% Determine the required output wordlength
	wout = wdata+ceil(log2(sum(abs(h))));
	
	% Generate and choose coefficient representation
	disp('Generating coefficient representation');
	creps = gencoeffreps(h, wout, signedcoeffs);
	crep = chooserep(creps, ctype, cid);
	
	% Compute the needed correction term from the coefficient and data formats
	cterm = calccterm(crep, wout, wdata, signeddata);
	
	% Create filter specification structure
	filtspec.h = h;
	filtspec.crep = crep;
	filtspec.cterm = cterm;
	filtspec.numphases = 1;
	filtspec.wdata = wdata;
	filtspec.wout = wout;
	filtspec.signeddata = signeddata;
	
	% Create partial product specification and map to input signals
	disp('Generating partial products');
	[ppspec, ppmap] = makeppgen(filtspec, archspec);
	
	% Create structure for partial product reduction tree
	% makewallace: Wallace tree
	% makedadda: Dadda tree
	% makera: Reduced Area heuristic
	disp('Generating CS tree');
	%csspec = makewallace(ppspec, archspec);
	%csspec = makedadda(ppspec, archspec);
	csspec = makera(ppspec, archspec);
	
	% Create a ripple-carry adder for the VMA
	disp('Generating VMA');
	vmaspec = makercvma(csspec, archspec);
	
	% Calculate the pipeline delay of the architecture
	piperegs = csdelay(csspec) + csdelay(vmaspec) - ppspec.strucdelay;
	
	% Print FIR information
	printfirinfo(filtspec, ppspec, pphw, csspec, vmaspec, piperegs);
	
	% Create the bit-level map of the reduction tree and VMA
	disp('Mapping CS cells');
	csmap = makecsmap(csspec);
	disp('Mapping VMA cells');
	vmamap = makecsmap(vmaspec);
	
	mkdir(vhdldir);
	
	disp('Generating VHDL');
	
	% Generate VHDL code for full adders, half adders, and registers
	genbasevhdl(vhdldir);
	
	% Generate bit-level FIR filter code
	genfirvhdl(vhdldir, vhdlname, filtspec, ppspec, ppmap, csspec, csmap, vmaspec, vmamap);
	
	% Generate high-level FIR direct form reference code
	firrefname = strcat(vhdlname, 'ref');
	genfirrefvhdl(vhdldir, firrefname, filtspec, piperegs);
	
	% Generate test bench for bit-level and high-level filters
	firtbname = strcat(vhdlname, '_tb');
	stiminfile = strcat(vhdlname, '_in.stim');
	stimoutfile = strcat(vhdlname, '_out.stim');
	genfirtbvhdl(vhdldir, firtbname, vhdlname, firrefname, stiminfile, stimoutfile, filtspec);
	
	% Generate stimuli files for testbench
	genfirstim(vhdldir, vhdlname, filtspec, 1024, piperegs);
	
	% Generate simulation script
	gensimscript(vhdldir);
	
	% Generate synthesis script
	gensynthscript(vhdldir);
	
